#include <math.h>
#include <alloc.h>

/*********************************************************/
/*                                                       */
/*                   Walsh Routines                      */
/*                                                       */
/*       Includes:   A fast Walsh Transform              */
/*                   An out-of-place Sort of increasing  */
/*                   wals                                */
/*                                                       */
/*       K. T. Kilty                       11/04/84      */
/*                                                       */
/*********************************************************/
/*   ==================================================  */
/*   A Fast Walsh Transform that leaves the transformed  */
/*   vector in a reverse bit gray code sequence.  A call */
/*   to sort_wals will order in increasing wals sequence */
/*   input variables are:  x - input data vector of n    */
/*                             elements (n a power of 2 )*/
/*                         y - contains the walsh trans. */
/*                             on output.                */
/*                         n - number of samples in x    */
/*   Read documentation in signal.txt for details        */
/*   =================================================== */

float  walsh(float x,int order)
{
int   q,j;
      if (order==0)
      {
	  if ((x<=1.0) && (x>0.0)) return(1.0); else return(0.0);
      }
      else
      {
	  q=order%2;
	  j=(order-q)/2;
	  return( walsh(2.0*x,j) +
	    (float)pow((double)(-1.0),(double)(j+q))*walsh(2.0*(x-0.5),j) );
      }
}  /* walsh evaluation */

float  hadamard(float x,int order)
{
float y;
int   q,j;
      if (order==0) return(1.0);
      else
	  if (order==1)
            if ((x>=0.0) && (x<=0.5)) return(1.0); else return(-1.0);
      q=(int)(order/2);
      j=order-2*q;
      y=2.0*x; if(y>1.0)y-=1.0;
      return( hadamard(y,q)*hadamard(x,j) );
}  /* hadamard evaluation */


void  fast_walsh(float *x,float *y,int n)
{
int   m,p,e=1,i,j,k,l,point;
float t;
/*   Calculation of first m-1 iterations of walsh trans. */
        m=(int)(3.321928*log10((double)n)+0.5); /* m=Log2(n) */
	p=m-1;
	for(l=0; l<p; l++)
	{
	   k=e;
	   e*=2;
	   for(j=0; j<k; j++)
	   {
	      for(i=j; i<n; i+=e)
	      {
		  point=i+k;
		  t=x[point];
		  x[point]=x[i]-t;
		  x[i]+=t;
	      }	/*	end loop on i	*/
	   }	/*	end loop on j	*/
	}	/*	end loop on l	*/
/*   calculations for last (mth) iteration of walsh tran */
/*   also switch resultant vector so that y contains the */
/*   walsh transform of the original x                   */
	l = n/2;
	for(j=0; j<l; j++)
	{
	   point=j+l;
	   t=x[point];
	   y[point]=x[j]-t;
	   y[j]=x[j]+t;
	}
}  /* fast_walsh */
/*   =================================================== */
/*   An out-of-place wals ordered sort using a locating  */
/*   vector generated in s, y contains walsh transform   */
/*   x will contain walsh transform ordered in increasing*/
/*   wals upon return                                    */
/*   =================================================== */
int  sort_wals(float *x,float *y,int n)
{
int   *s,m,p,e=2,i,j,k,l;
/*   The locating vector is generated to a length of     */
/*   n/2 through the first m-1 iterations below          */
        m=(int)(3.321928*log10((double)n)+0.5); /* m=Log2(n) */
	s=(int*)calloc((unsigned)(n/2),sizeof(int));
	if (s==NULL) return(-1);
	s[0]=0;
	s[1]=1;
	p=m-1;
	for(l=1; l<p; l++)
	{
	   k=e;
	   e*=2;
	   for(j=0; j<k; j++)
	   {
	      s[j]*=2;
	      s[e-j-1]=s[j]+1;
	   } 	/*	loop on j	*/
	}	/*	loop on l	*/
/*   In the last iteration generate the entire locating   */
/*   vector and re-order the walsh transform in y to x    */
	p=n/2;
	for(k=0; k<p; k++)
	{
	   j=2*s[k];
	   x[k]=y[j++];
	   x[n-k-1]=y[j];
	}
	free(s);
	return(0);
}   /* sort_wals */

/*   =================================================== */
/*   This routine reverts the ordered sort of increasing */
/*   wals into a gray code sequence.  The purpose is to  */
/*   effect an inverse walsh transform after a filter    */
/*   operation.  The sort vector is generated in s.      */
/*   y is the input vector.  x is the output vector.     */
/*   =================================================== */
int  gray_sort(float *x,float *y,int n)
{
int   *s,m,p,e=2,i,j,k,l;
/*   The locating vector is generated to a length of     */
/*   n/2 through the first m-1 iterations below          */
        m=(int)(3.321928*log10((double)n)+0.5); /* m=Log2(n) */
	s=(int*)calloc((unsigned)(n/2),sizeof(int));
	if (s==NULL) return(-1);
	s[0]=0;
	s[1]=1;
	p=m-1;
	for(l=1; l<p; l++)
	{
	   k=e;
	   e*=2;
	   for(j=0; j<k; j++)
	   {
	      s[j]*=2;
	      s[e-j-1]=s[j]+1;
	   } 	/*	loop on j	*/
	}	/*	loop on l	*/
/*   In the last iteration generate the entire locating   */
/*   vector and re-order the walsh transform in y to x    */
	p=n/2;
	for(k=0; k<p; k++)
	{
	   j=2*s[k];
	   x[j++]=y[k];
	   x[j]=y[n-k-1];
	/*   x[k]=y[j++]; x[n-k-1]=y[j]; */
	}
	free(s);
	return(0);
}   /* gray_sort */

