#define    NIL      -1
#define    TRUE     1
#define    FALSE    0
#define    TWO_PI   6.283185

#include   <alloc.h>
#include   <math.h>

/*
      ============================
             MEDIAN  FILTER
      ============================

      The user passes floating point data
      in array (a), and the length of this
      vector (num).  The desired length
      of the median filter is supplied
      in the integer 'length', this must
      be an odd number.

      The median sort routine first sorts
      a series of 'length' data values in
      decending order in the module 'insert_sort'.
      For each data point between length/2 and
      num-length/2-1 the routine inserts a new
      value of the time series into the list
      with the insertion sort 'insert'.  The
      algorithm runs in a number of operations
      proportional to length.  A complete
      sort of the data within the filter span at
      each step would run in proportion to
      (length)^2.

      K. Anderson of Lincoln Labs wrote a median
      filter that also runs in a number of steps
      proportional to (length) and which is
      included in a USGS open file report 81-1091
      His routine shuffles input values in a
      stack that always contains the median in
      the central position.  It may be a little
      faster than this routine.  I don't know.

      Global items include all those dealing
      with the sorted lists.

      *Sort_List is an array of index values
      back into the data array (a) for
      values with the current filter aperture.

      *Link_Down points to an array of integers
      that make up pointers from the largest
      to smallest data values within the current
      filter aperture.

      *Link_Up  is an array of integer pointers
      from smallest to largest data value within
      the filter aperture.

      *Link_Time is an array of integer pointers
      from oldest data value within the aperture
      to newest.  This allows the algorithm to
      determine which data value to drop.

      get_free() returns the only available
      array element for storing the next data
      value.

       k.t.kilty

*/

/* GLOBAL */

int   *Sort_List,*Link_Up,*Link_Down,*Link_Time;
int   Oldest,Newest,First,Last;

int   get_free(void)
{
int   new;
      /* get oldest element */
      new=Oldest;
      Oldest=Link_Time[Oldest];
      Link_Time[Newest]=new;
      if (new==First) First=Link_Down[First];
      else Link_Down[Link_Up[new]]=Link_Down[new];

      if (new==Last) Last=Link_Up[new];
      else Link_Up[Link_Down[new]]=Link_Up[new];

      Newest=new;
      return(new);
}  /* get_free */


float  insert(float *a,int current,int middle)
{
/*  this routine does insertion sorting on the fly */
int    new,next;
float  test;
       new=get_free();
       next=First;
       test=a[current];
       if (test>=a[Sort_List[next]] )  /* new item is first on list */
       {
           Link_Up[First]=new;
	   Link_Down[new]=First;
	   Link_Up[new]=NIL;
	   First=new;
       }
       else
       { /* new element is farther into list */
          while( (next=Link_Down[next])!=NIL )
	  {
	     if (test>=a[Sort_List[next]] )
	     {  /* found correct position -- alter links */
	        Link_Down[new]=next;
		Link_Up[new]=Link_Up[next];
	        Link_Down[Link_Up[next]]=new;
                Link_Up[next]=new;
	        break;
	     }
	  }
       } /* if structure */
       /* if next=NIL new value goes at end of list */
       if (next==NIL)
       {
	  Link_Down[new]=NIL;
	  Link_Up[new]=Last;
	  Link_Down[Last]=new;
	  Last=new;
       }
       Sort_List[new]=current;

       /* find median value to return */
       next=First;
       for(new=0; new<middle; new++) next=Link_Down[next];

       return(a[Sort_List[next]]);
} /* insert */

float  insert_sort(float *a,int num,int median)
{
/* this routine is called once to sort the original list
   the remainder of sorting is done on the fly by the routine
   named insert */
int    i,next;
float  test;
       Link_Down[0]=NIL;
       Last=First=0;
       for(i=1; i<num; i++)
       {
          /* set up initial links in these lists as we go */
	  Link_Time[i-1]=Sort_List[i]=i;
	  Link_Down[i]=i+1;
	  next=First;

	  /* get a test value from next item in time series */
	  test=a[Sort_List[i]];
	  if (test>=a[Sort_List[next]]) /* new element fits in first */
	  {
	     Link_Down[i]=First;
	     Link_Up[First]=i;
	     Link_Up[i]=NIL;
	     First=i;
	  } /* new element becomes first */
	  else
	  { /* new element is farther into list */
	     while( (next=Link_Down[next])!=NIL)
	     {
	        if (test>=a[Sort_List[next]] )
	        { /* found correct position -- alter links */
	           Link_Down[i]=next;
	  	   Link_Up[i]=Link_Up[next];
	           Link_Down[Link_Up[next]]=i;
                   Link_Up[next]=i;
	           break;
	        }
  	     } /* while loop to link */
	     if (next==NIL)
	     { /* new element belongs at end of list */
	        Link_Down[Last]=i;
	        Link_Down[i]=NIL;
		Link_Up[i]=Last;
		Last=i;
	     } /* new item is last on list */
	  } /* if element is farther into list */
       } /* main 'for' loop to build list */

       /* set up pointers to newest and oldest elements in time series */
       Oldest=0; Newest=num-1; Link_Time[Newest]=NIL;

       /* look for median value to return */
       next=First;
       for(i=0; i<median; i++) next=Link_Down[next];
       return(a[Sort_List[next]]);
} /* insert_sort */

int    median_filter(float *a,int num,int length)
{
int    last,median,i;
float  median_value;
       /* allocate space for linked list */
       if ((length%2)==0) return(-1);
       Sort_List=(int*)calloc((unsigned)length,sizeof(int));
       if (Sort_List==NULL) return(-2);
       Link_Down=(int*)calloc((unsigned)length,sizeof(int));
       if (Link_Down==NULL) return(-2);
       Link_Up  =(int*)calloc((unsigned)length,sizeof(int));
       if (Link_Up==NULL)   return(-2);
       Link_Time=(int*)calloc((unsigned)length,sizeof(int));
       if (Link_Time==NULL) return(-2);
       median=length/2;  /* median value pointer and first data point filtered */
       last=num-median-1; /* last point in input to be filtered */

       /* set up the linked list for sorting on the fly */
       median_value=insert_sort(a,length,median);

       for(i=median; i<last; i++)
       {
	  a[i]=median_value;
	  median_value=insert(a,i+median+1,median);
       }  /* filtering loop */
       a[last]=median_value;

       free(Link_Time);
       free(Link_Up);
       free(Link_Down);
       free(Sort_List);

       return(0);
} /* median filter */

