/*
      -----------------------------------
	   LOMB-SCARGLE PERIODOGRAM
      -----------------------------------

      This routine accepts an array of n times
      (t) at which n values (v) are observed.
      The user supplies arrays x and y which
      return frequency and the lomb-scargle
      normalized periodogram value at a series
      of m locations.  m is equal to the
      product  n*sample/2.  m is calculated
      by the user and supplied to the routine.
      The user should have allocated m storage
      locations for f and p.  The
      return value from the routine is the
      probability that the maximum value of
      the periodogram, pointed to by the
      value of largest (returned), is the
      result of random noise.

      n=number of input pairs
      sample=over sampling factor (>=1)
      f=radian frequency

      The average nyquist frequency is equal to
      n/(2*delta_t) where delta_t is the
      total time range of the time series,
      t[n]-t[0].

      k.t. kilty
*/

#define    TWO_PI  6.283185307

#include   <alloc.h>
#include   <math.h>


float lomb_scargle(float *t,float *v,int n,float *f,float *p,int m,
		   int sample,int *largest)
{
int    i,j,total;
double *work_r,*work_i,*work_pi,*work_pr;
double work,argument,power,expo;
float  mean,variance,sin_tau,cos_tau,sum_cv,sum_sv,sum_c,sum_s,s,c;
float  largest_p=0.0,tau,start,t_min,t_max,delta_t,t_ave,delta_p;
float  probability;
      total=n*sample/2;
      if (total>m) return(-1.0);
      work_r= (double*)calloc((unsigned)n,sizeof(double));
      if (work_r==NULL) return(-2.0);
      work_i= (double*)calloc((unsigned)n,sizeof(double));
      if (work_i==NULL) return(-2.0);
      work_pr=(double*)calloc((unsigned)n,sizeof(double));
      if (work_pr==NULL) return(-2.0);
      work_pi=(double*)calloc((unsigned)n,sizeof(double));
      if (work_pi==NULL) return(-2.0);
      mean=variance=0.0; t_max=t_min=t[0];
      for(i=0; i<n; i++)
      {
	 mean+=v[i];
	 variance+=v[i]*v[i];
	 if (t_max<t[i]) t_max=t[i];
	 else if (t_min>t[i]) t_min=t[i];
      } /* loop over i (data) */
      mean/=n;
      variance=(variance-n*mean*mean)/(float)(n-1);
      delta_t=t_max-t_min;
      t_ave=0.5*(t_max+t_min);
      start=delta_p=1.0/(delta_t);
      for(i=0; i<n; i++)
      {
	 argument=TWO_PI*start*(t[i]-t_ave);
	 work=sin(0.5*argument);
	 work_pr[i]=-2.0*work*work;
	 work_i[i]=work_pi[i]=sin(argument);
	 work_r[i]=cos(argument);
      } /* loop over  i */
      for(i=0; i<total; i++)
      {
	 f[i]=start;
	 sum_sv=sum_c=0.0;
	 for(j=0; j<n; j++)
	 {
	    sum_sv+=work_r[j]*work_i[j];
	    sum_c+=(work_r[j]-work_i[j])/(work_r[j]+work_i[j]);
	 } /* loop over j to calc tau */
	 tau=0.5*atan2( (double)2.0*sum_sv,(double)sum_c);
	 sin_tau=sin((double)tau);
	 cos_tau=cos((double)tau);
	 sum_s=sum_c=sum_sv=sum_cv=0.0;
	 for(j=0; j<n; j++)
	 {
	    s = work_i[j]*cos_tau-work_r[j]*sin_tau; sum_s+=s*s;
	    c = work_r[j]*cos_tau+work_i[j]*sin_tau; sum_c+=c*c;
	    work = v[j]-mean;
	    sum_sv+=s*work;
	    sum_cv+=c*work;
	    work = work_r[j];
	    work_r[j]+=(work_r[j]*work_pr[j]-work_i[j]*work_pi[j]);
	    work_i[j]+=(work_i[j]*work_pr[j]+     work*work_pi[j]);
	 } /* loop over j for periodogram val */
         p[i]=0.5*((sum_cv*sum_cv/sum_c)+(sum_sv*sum_sv/sum_s))/variance;
	 if (p[i]>largest_p) { largest_p=p[i]; *largest=i; }
	 start+=delta_p;
      } /* main loop over i */

      free(work_pi);
      free(work_pr);
      free(work_i);
      free(work_r);

      power=2.0*(float)total/(float)sample;
      expo=exp(-(double)largest_p);
      probability=power*expo;
      if (probability>0.01) probability=1.0-pow((1.0-expo),power);
      return(probability);
} /* lomb-scargle */

#include <stdio.h>
#include <time.h>
#include <stdlib.h>
#define OMEGA 0.628

int fcmp(float *a,float *b)
{
    if (*a<*b) return(-1);
    else if (*a==*b) return(0);
    else return(1);
}

main()
{
FILE *out;
float pr,sigma;
float t[100],v[100]; /* test arrays */
float *f,*p;
int maximum;
int sample=2;
int m,i,n=100;
int largest;

out = fopen("lomb.dat","w");

maximum = (int)((float)((n/2)*sample)/10.0); /* multiple of the average nyquist rate */
m = n*sample/2;

f = (float*)calloc(m,sizeof(float));
p = (float*)calloc(m,sizeof(float));

/* make a random series with 3 real components plus random noise */
randomize();
for(i=0; i<n; i++)
{
   t[i]=10.0*(float)rand()/32767.0;
}
/* sort these random times */
qsort(t,n,sizeof(float),fcmp);
for(i=0; i<n; i++)
{
   sigma=t[i]*OMEGA;
   v[i]=5.0*cos(3.0*sigma)+7.0*sin(60.0*sigma)+5.0*cos(10.*sigma)+((float)rand()-16383.0)/16383.0;
}

pr = lomb_scargle(t,v,n,f,p,m,sample,&largest);
fprintf(out,"Largest value occurs at %4i with probability %10.4f\n",largest,pr);
for(i=0; i<m; i++)
  fprintf(out,"%4i freq=%10.4f mag=%10.4f\n",i,f[i],p[i]);
}